unit tdirscan;

{
   TDirectoryScan

   A Delphi - compatible VCL control which encapsulates the recursive scanning
   of file system directories in Win16 or Win32 file systems.

   An original work, Copyright 1997 by Donald L. Wallace. The right is granted
   to use this source code in any software program as long as this copyright
   notice and these comments are included with the source code or any source
   code derived from this file. No warranty of fitness for any purpose is
   provided. Use at your own risk.

   Author Email address: 71247.3221@Compuserve.com
}


interface

uses
  Winprocs, WinTypes, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TVerifyFile = Procedure(Sender : TObject;path : String;
                          stream : TFileStream;TimeStamp : TDateTime;
                          var PassFlag : Boolean) of Object;
  TProcessFile =  procedure(Sender : TObject;path : String) of Object;
  TDirList = procedure(Sender : TObject;paths : TStrings) of Object;
  TDirectoryScan = class(TComponent)
  private
    FCanceled : Boolean;
    FOnVerifyFile : TVerifyFile;
    FDirPath,FWildCard : String;
    FProcessing : Boolean;
    FProcessFile : TProcessFile;
    FProcessDirectory : TProcessFile;
    FOnDirList : TDirList;
    FOnProgress : TNotifyEvent;
    FCurDirectory : String;
    FTotalCount : Integer;
    FSubdirectories : Boolean;
    procedure ProcessFile(path : String);
    procedure ProcessDirectory(path : String);
    Function FileAcceptable(path : String) : Boolean;
    { Private declarations }
  protected
    { Protected declarations }
  public
    { Public declarations }
    Procedure Cancel;
    Function Execute : Boolean;
    Constructor Create(Owner : TComponent); Override;
  published
    { Published declarations }
    property Directory : String Read FDirPath Write FDirPath;
    property CurDirectory : String Read FCurDirectory;
    property TotalCount : Integer Read FTotalCount;
    property FileName : String Read FWildCard Write FWildCard;
    property OnVerifyFile : TVerifyFile Read FOnVerifyFile Write FOnVerifyFile;
    property OnProcessFile : TProcessFile Read FProcessFile Write FProcessFile;
    property OnDirList : TDirList Read FOnDirList Write FOnDirList;
    property OnProcessDirectory : TProcessFile Read FProcessDirectory Write FProcessDirectory;
    property OnProgress : TNotifyEvent Read FOnProgress Write FOnProgress;
    property Subdirectories : Boolean Read FSubdirectories Write FSubdirectories Default True;
  end;

procedure Register;

implementation

Function TDirectoryScan.FileAcceptable(path : String) : Boolean;
var FileStream : TFileStream;
    FileDateStamp : TDateTime;
Begin
   if not Assigned(FOnVerifyFile) then
      Result := True
   else
   begin
      try
         FileStream := TFileStream.Create(path, fmOpenread or fmShareDenyNone);
      except
         FileStream := nil;
      end;
      if (FileStream <> nil) and (FileStream.Handle > 0) then
      begin
         Result := True;
         FileDateStamp := FileDateToDateTime(FileGetDate(FileStream.Handle));
         FOnVerifyFile(Self,path,FileStream,FileDateStamp,Result);
      End
      else
         Result := False;
      FileStream.Free;
   end;
End;

{$IFNDEF WIN32}
Function Trim(const s : String) : String;
var b,e : Integer;
Begin
   b := 1;
   e := Length(s);
   while (s[b] = ' ') and (b <= Length(s)) do
      Inc(b);
   while (s[e] = ' ') and (e >= 0) do
      Dec(e);
   Result := Copy(s,b,1+e-b);
End;
{$ENDIF}

Function TDirectoryScan.Execute : Boolean;
{
   Perform the scan of the directory tree. Returns TRUE if at least one
   matching file was located, FALSE if no files were located.
}
var sr : TSearchRec;
    ret : Integer;
    pathspec : String;
    Count : Integer;
    paths : TStringList;
    i : Integer;
    wildcard,dirpath : String;
begin

   { Reject reentrant calls into Execute }
   if FProcessing then
   begin
      Result := False;
      Exit;
   end;

   try

   FCurDirectory := '';
   FTotalCount := 0;
   FCanceled := False;
   FProcessing := True;
   wildcard := Trim(FWildcard);
   dirpath := Trim(FDirpath);
   if dirpath[Length(dirpath)] = '\' then
      dirpath := copy(dirpath,1,Length(dirpath)-1);

   if wildcard = '' then
      wildcard := '*.*';

   paths := TStringList.Create;
   paths.Add(dirpath);

   {
      Build a list of all directories and subdirectories contained
      in the passed directory specification. The list is built so that
      subdirectories immediately follow their parent directories in the list.
   }
   i := 0;
   if FSubdirectories then
   while i < paths.Count do
   begin
      dirpath := paths[i];
      Inc(i);
      pathspec := dirpath + '\*.*';
      FCurDirectory := dirpath;
      ret := FindFirst(pathspec,faDirectory,sr);
      while ret = 0 do
      begin
         Application.ProcessMessages;
         if Assigned(FOnProgress) then
            FOnProgress(Self);
         if FCanceled then
            Break;
         if ((sr.Attr and faDirectory) <> 0) and
            (sr.name <> '.') and
            (sr.name <> '..')  then
            begin
{$IFNDEF WIN32} {For Delphi 1, don't bother with over long paths }
               if Length(dirpath) + Length(sr.name) + 1 < 255 then
{$ENDIF}
               paths.Insert(i,dirpath + '\' + sr.name);
            end;
         ret := FindNext(sr);
      end;
      FindClose(sr);
      if FCanceled then
         Break;
   end;

   {
      If the user has assigned an event handler for the candidate list of
      directories, then call it.
   }
   if Assigned(FOnDirList) then
      FOnDirList(Self,paths);

   {
      Scan each subdirectory in list of directories we just built for
      matching files.
   }
   for i := 0 to paths.Count-1 do
   begin
      Count := 0;
      dirpath := paths[i];
      pathspec := dirpath + '\' + wildcard;
      FCurDirectory := dirpath;
      ret := FindFirst(pathspec,faAnyFile and (not faDirectory),sr);
      while ret = 0 do
      begin
         Application.ProcessMessages;
         if Assigned(FOnProgress) then
            FOnProgress(Self);
         if FCanceled then
            Break;
         if ((sr.Attr and faDirectory) = 0) and
            FileAcceptable(dirpath + '\' + sr.name) then
         begin
            if Count = 0 then
               ProcessDirectory(dirpath);
            ProcessFile(dirpath + '\' + sr.name);
            Inc(Count);
            Inc(FTotalCount);
         end;
         ret := FindNext(sr);
      end;
      FindClose(sr);
      if FCanceled then
         Break;
   end;

   except
      on E : Exception do
         ShowMessage('Error during processing');

   end;

   paths.Free;

   FProcessing := False;

   Result := (FTotalCount > 0);

   if Result then
   begin
      FCurDirectory := '';
      if Assigned(FOnProgress) then
         FOnProgress(Self);
   end;
end;

Constructor TDirectoryScan.Create(Owner : TComponent);
Begin
   Inherited Create(Owner);
   FSubdirectories := True;
End;

procedure TDirectoryScan.ProcessFile(path : String);
Begin
   if Assigned(FProcessFile) then
      FProcessFile(Self,path);
End;

procedure TDirectoryScan.ProcessDirectory(path : String);
Begin
   if Assigned(FProcessDirectory) then
      FProcessDirectory(Self,path);
End;

Procedure TDirectoryScan.Cancel;
Begin
   FCanceled := True;
End;

procedure Register;
begin
  RegisterComponents('Samples', [TDirectoryScan]);
end;

end.
